import { randomBytes } from 'crypto'
import { H3Event, getQuery } from 'h3'

export function generateSecret(byteLength: number = 32): string {
  return randomBytes(byteLength).toString('hex')
}

// 发起登录请求的状态参数结构
export interface OAuthState {
  state: string
  timestamp: number
  userAgent: string
}

// 创建状态数据
export async function createOAuthState(event: H3Event): Promise<OAuthState> {
  const userAgent = getRequestHeader(event, 'user-agent') || ''

  const stateData = {
    state: generateSecret(),
    timestamp: Date.now(),
    userAgent
  } as OAuthState

  await useStorage().setItem(`oauth:${stateData.state}`, stateData)

  return stateData.state
}

// 验证状态数据
export async function validateOAuthState(event: H3Event) {
  const { state } = getQuery(event)

  const storage = useStorage()

  if (!state || typeof state !== 'string') {
    throw createError({
      statusCode: 400,
      statusMessage: '不合理的state'
    })
  }

  // 从 storage 中检索存储的状态
  const storedState = await storage.getItem<OAuthState>(`oauth:${state}`)

  if (!storedState) {
    throw createError({
      statusCode: 400,
      statusMessage: '不合法的state'
    })
  }

  // 验证时间戳
  const currentTime = Date.now()
  if (currentTime - storedState.timestamp > 600000) { // 10分钟超时
    await storage.removeItem(`oauth:${state}`)
    throw createError({
      statusCode: 400,
      statusMessage: '单点登录请求超时'
    })
  }

  // 验证用户代理
  const userAgent = getRequestHeader(event, 'user-agent') || ''
  console.log('userAgent', userAgent, storedState.userAgent)
  if (userAgent !== storedState.userAgent) {
    throw createError({
      statusCode: 400,
      statusMessage: 'User agent 不匹配'
    })
  }

  // 清除存储的状态
  await storage.removeItem(`oauth:${state}`)
}