import { H3Event } from 'h3'
import jwt from 'jsonwebtoken'
import { useRuntimeConfig } from '#imports'
import { validateOAuthState } from '~/server/utils/state'

export default defineEventHandler(async (event: H3Event) => {
  const config = useRuntimeConfig()
  const { prisma } = event.context
  const { code } = getQuery(event)

  // 验证state
  await validateOAuthState(event)

  // 发起请求获取accessToken
  const result = await $fetch(`${config.oauth.endpoint}/oauth/token`, {
    method: 'POST',
    "Content-Type": 'application/json',
    body: {
      grant_type: 'authorization_code',
      client_id: config.oauth.clientId,
      client_secret: config.oauth.clientSecret,
      code,
    }
  })

  // result 结构： access_token, refresh_token, expires_in, id_token, token_type: 'Bearer'
  // access_token 是用来请求服务平台各资源接口的授权凭证，放在请求头部，可持久化存储在本地服务中
  // id_token 是一个含有用户基本信息的 json_web_token, 可通过 jsonwebtoken 库验证并解出其中携带的基本用户信息,用于创建本方服务的用户信息
  const decodeIdToken = jwt.decode(result.id_token)

  if (!decodeIdToken) {
    throw createError({
      statusCode: 400,
      statusMessage: 'idToken解析失败'
    })
  }

  // 解出结构示例：
  // {
  //  "iss": "dtbim.cn", // token颁发者
  //  "sub": "U4tFkMF5RtpJ", // 用户id
  //  "aud": "OC6DTPjeuSIHQ5CnENm0", // oauth client_id
  //  "exp": 1726551274, // 过期时间，秒级unix时间戳
  //  "iat": 1725946474, // token签发时间，秒级unix时间戳
  //  "auth_time": 1725946474, // 用户身份认证的时间，秒级unix时间戳
  //  "name": "root", // 当前认证的用户的用户名
  //  "email": "root@myinkslab.com", // 当前认证的用户邮箱
  //  "picture": null // 当前认证的g用户的个人头像
  // }

  // 依据上述信息创建或更新当前系统的用户表
  const user = await prisma.user.upsert({
    where: {
      email: decodeIdToken.email,
    },
    create: {
      email: decodeIdToken.email,
      name: decodeIdToken.name,
      user_id: decodeIdToken.sub,
      picture: decodeIdToken.picture,
      refresh_token: result.refresh_token,
      access_token: result.access_token,
      expires_at: decodeIdToken.exp,
      token_type: result.token_type,
      scope: decodeIdToken.scope,
      id_token: result.id_token,
    },
    update: {
      name: decodeIdToken.name,
      user_id: decodeIdToken.sub,
      picture: decodeIdToken.picture,
      refresh_token: result.refresh_token,
      access_token: result.access_token,
      expires_at: decodeIdToken.exp,
      token_type: result.token_type,
      scope: decodeIdToken.scope,
      id_token: result.id_token,
      updated_at: new Date(),
    }
  })

  // 根据创建的用户信息，生成cookie记录，形成当前服务的登录态
  const generatedToken = jwt.sign({
    user_id: user.id,
    user_name: user.name,
    email: user.email,
    picture: user.picture,
  }, config.jwt.secret, { expiresIn: config.jwt.ttl })
  // 写入cookie
  setCookie(event, 'authToken', generatedToken, {
    path: '/',
    secure: process.env.NODE_ENV === 'production',
    maxAge: config.jwt.ttl
  })

  // 回到登录首页
  return await sendRedirect(event, '/', 302)
})